<?php
/**
 * CakePHP(tm) : Rapid Development Framework (https://cakephp.org)
 * Copyright (c) Cake Software Foundation, Inc. (https://cakefoundation.org)
 *
 * Licensed under The MIT License
 * For full copyright and license information, please see the LICENSE.txt
 * Redistributions of files must retain the above copyright notice.
 *
 * @copyright Copyright (c) Cake Software Foundation, Inc. (https://cakefoundation.org)
 * @link      https://cakephp.org CakePHP(tm) Project
 * @since     0.2.9
 * @license   https://opensource.org/licenses/mit-license.php MIT License
 */
namespace App\Controller;

use Cake\Http\Exception\NotFoundException;

/**
 * Static content controller
 *
 * This controller will render views from Template/Home/
 *
 */
class ContentController extends AppController
{
  /**
   * Initialization hook method.
   *
   * Use this method to add common initialization code like loading components.
   *
   * e.g. `$this->loadComponent('Security');`
   *
   * @return void
   */
  public function initialize(): void
  {
    parent::initialize();

    $this->PageContents = $this->fetchTable('PageContents');
    $this->Banners = $this->fetchTable('Banners');
    $this->Files = $this->fetchTable('Files');
    $this->Galleries = $this->fetchTable('Galleries');
  }

  /*
   * Main method for home of the main menu clicks
   * Handles loading of contents for rendering on section homes
   */
  public function index()
  {
    /** 
     * Default home for basic section features
     *
     * The selected menu is in the section param of the request
     * Lets retreive the data and load the content for that section
     * 
     */

    // Retrieve the menu using the section (Should be the same as the menu slug)
    $menuDetails = $this->PageContents->Menus->find('all', [
      'conditions' => [
        'slug' => $this->request->getParam('section')
      ]
    ])->first();

    /**
     * Only load the content if the menu is active, else log the issue and throw a not found exception
     * And ofcourse, if no menu is found with the section slug, then there is no need to continue as well
     */
    if ($menuDetails == null) {
      $this->log('No menu found with the selected slug: ' . $this->request->getParam('section'), 'debug');
      throw new NotFoundException();
    } else if ($menuDetails->status != 1) {
      $this->log('Menu found with slug: ' . $this->request->getParam('section') . ' but not active', 'debug');
      throw new NotFoundException();
    }

    // Initialize the content array and populate the contents of each section
    $viewContent = $this->PageContents->Contents->find('all', [
      'contain' => [
        'ContentImages',
        'PageContents'
      ],
      'conditions' => [
        'status' => 1
      ]
    ])->order([
          'Contents.order_value' => 'ASC'
        ])->matching('PageContents', function ($q) use ($menuDetails) {
          return $q->where(['PageContents.menu_id' => $menuDetails->id]);
        })->toArray();

    // Load the banners. Banners are normally loaded at the top of the page.
    // So lets add it to the sections.
    // Append to any section if you want to show as well, could be carousel, etc....
    $banners = $this->Banners->find('all', [
      'conditions' => [
        'status' => 5
      ]
    ])
      ->order(['rand()'])
      ->limit(1)
      ->toArray();

    // Load the files for the page.
    // For now lets load all the files, 
    // we will show it on the relevant pages.
    // TODO: Fix the file associations for the menus

    // $files = $this->Files->find('all', [
    // 	'conditions' => [
    // 		'status' => 1
    // 	]
    // ])->order(['date_created' => 'DESC'])->toArray();

    $files = $this->Files->find('all', [
      'conditions' => [
        'status' => 1
      ]
    ])->order(['date_created' => 'DESC'])->toArray();
    //annual reports and downloads


    $otherDownloads = $this->Files->find('all', [
      'conditions' => [
        'status' => 1
      ]
    ])
      ->where(['title NOT LIKE' => '%Annual Report%'])
      ->order(['date_created' => 'DESC'])->toArray();

    // Load details on sub menus list
    $subMenuLists = $this->PageContents->Menus->find('all', [
      'conditions' => [
        // 'status' => '1',
        'parent_menu_id' => $menuDetails->id
      ]
    ])->toArray();

    /**
     * Perform additional task here before sending the values
     * to be rendered on the view
     */

    $this->set(compact('viewContent', 'menuDetails', 'banners', 'files', 'otherDownloads', 'subMenuLists'));
  }

  /*
   * Handles the event of clicking a sub menu
   * Performs generic actions to retrieve required content of that sub menu
   */
  public function subsection()
  {
    /** 
     * Default sub section for basic section features with sub menus
     *
     * The selected sub menu is in the subsection param of the request
     * The selected main menu is in the section param of the request
     * 
     * Lets retreive the data and load the content for that sub section
     * 
     */

    // Retrieve the sub menu using the section (Should be the same as the menu slug)
    $menuDetails = $this->PageContents->Menus->find('all', [
      'conditions' => [
        'slug' => $this->request->getParam('subsection')
      ]
    ])->first();

    /**
     * Only load the content if the menu is active, else log the issue and throw a not found exception
     * And ofcourse, if no menu is found with the section slug, then there is no need to continue as well
     */
    if ($menuDetails == null) {
      $this->log('No menu found with the selected slug: ' . $this->request->getParam('section'), 'debug');
      throw new NotFoundException();
    } else if ($menuDetails->status != 1) {
      $this->log('Menu found with slug: ' . $this->request->getParam('section') . ' but not active', 'debug');
      throw new NotFoundException();
    }

    //Load product submenu content
    $submenus = $this->PageContents->Menus->find('all', [
      'conditions' => [
        'parent_menu_id' => $menuDetails->id
      ]
    ])->select(['id']);
    $condition = ['PageContents.menu_id' => $menuDetails->id];
    if ($submenus->count() > 0) {
      $condition = [
        'OR' => [
          ['PageContents.menu_id' => $menuDetails->id],
          ['PageContents.menu_id in' => $submenus]
        ]
      ];
    }

    // Initialize the content array and populate the contents of each section
    $viewContent = $this->PageContents->Contents->find('all', [
      'contain' => [
        'ContentImages',
        'PageContents'
      ],
      'conditions' => [
        'status' => 1
      ]
    ])->order([
          'Contents.order_value' => 'ASC'
        ])->matching('PageContents', function ($q) use ($condition) {
          return $q->where($condition);
        })->toArray();

    /**
     * Perform additional task here before sending the values
     * to be rendered on the view
     */
    // Retrieve the submenus of this menu
    // We will use this to display sub menus in the form of category page / view
    $categoryContents = $this->PageContents->Menus->find('all', [
      'contain' => [
        'ParentMenus'
      ],
      'conditions' => [
        'Menus.parent_menu_id' => $menuDetails->id,
        'Menus.status' => 1
      ]
    ]);

    // Initialize the content array and populate the contents of each section
    $homeAlbums = $this->Galleries->Albums->find('all', [
      'contain' => [
        'Galleries'
      ],
      'conditions' => [
        'Albums.status' => 1,
        'Albums.show_on_home' => 1
      ]
    ])->toArray();

    // Load the banners. Banners are normally loaded at the top of the page.
    // So lets add it to the sub sections.
    // Append to any section if you want to show as well, could be carousel, etc....
    $banners = $this->Banners->find('all', [
      'conditions' => [
        'status' => 5
      ]
    ])
      ->order(['rand()'])
      ->limit(1)
      ->toArray();


    // Load details on sub menus list
    $subMenuLists = $this->PageContents->Menus->find('all', [
      'conditions' => [
        'status' => '1',
        'parent_menu_id' => $menuDetails->id
      ]
    ]);

    // Load the files for the page.
    // For now lets load all the files, 
    // we will show it on the relevant pages.
    // TODO: Fix the file associations for the menus
    $files = $this->Files->find('all', [
      'conditions' => [
        'status' => 1
      ]
    ])
      ->toArray();
    $annualReports = $this->Files->find('all', [
      'conditions' => [
        'status' => 1
      ]
    ])->where(['title LIKE' => '%Annual Report%'])
      ->order(['date_created' => 'DESC'])->toArray();
    $this->set(compact('viewContent', 'menuDetails', 'categoryContents', 'banners', 'subMenuLists', 'files', 'annualReports', 'homeAlbums'));

    // Render the same view for the main menu
    // Most views are similar accross the site
    $this->render('sub_section');
  }

  /**
   * Handles the details of content
   * Retrive the content details 
   * and match the menu to compare if it is the right link
   */
  public function details($id = null)
  {
    /** 
     * The selected menu is in the section param of the request
     * 
     * Lets retreive the data and load the content for that sub section
     * 
     */

    // Retrieve the sub menu using the section (Should be the same as the menu slug)
    $menuDetails = $this->PageContents->Menus->find('all', [
      'conditions' => [
        'slug' => $this->request->getParam('section')
      ]
    ])->first();

    /**
     * Only load the content if the menu is active, else log the issue and throw a not found exception
     * And ofcourse, if no menu is found with the section slug, then there is no need to continue as well
     */
    if ($menuDetails == null) {
      $this->log('No menu found with the selected slug: ' . $this->request->getParam('section'), 'debug');
      throw new NotFoundException();
    } else if ($menuDetails->status != 1) {
      $this->log('Menu found with slug: ' . $this->request->getParam('section') . ' but not active', 'debug');
      throw new NotFoundException();
    }

    // Get the full details of the content with the id
    $viewDetails = $this->PageContents->Contents->find('all', [
      'contain' => [
        'ContentImages',
        'PageContents'
      ],
      'conditions' => [
        'Contents.status' => 1,
        'Contents.id' => $id
      ]
    ])->first();

    $getContentDetails = $this->PageContents->Contents->find('all')
      ->contain(['ContentImages'])
      ->matching('PageContents', function ($q) use ($id) {
        return $q->where(['PageContents.menu_id' => $id]);
      })
      ->where(['Contents.status' => 1])
      // ->order(['Contents.order_value' => 'ASC'])
      ->toArray();

      $getMenuDetailsContent = $this->PageContents->Menus->find('all', [
        'conditions' => [
          'id' => $id,
        ]
      ])->first();

    if ($viewDetails == null) {
      $this->log('No details found for content with ID: ' . $id, 'debug');
      throw new NotFoundException();
    }

    /**
     * Perform additional task here before sending the values
     * to be rendered on the view
     */
    $banners = $this->Banners->find('all', [
      'conditions' => [
        'status' => 5
      ]
    ])
      ->order(['rand()'])
      ->limit(1)
      ->toArray();
    /**
     * Set the final values to the view
     */
    $this->set(compact('viewDetails', 'menuDetails', 'banners', 'getContentDetails','getMenuDetailsContent'));
  }
}
